//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for cursor changing demo.
//				This file is used for BOTH MacOS and Windows.
//
//	Written by:	Tim Monroe
//				Based (heavily!) on the MovieShell code written by Apple DTS.
//
//	Copyright:	 1994-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <11>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <10>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu identifiers
//	   <9>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <8>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <7>	 	06/27/97	rtm		added code to change some hot spot cursors
//	   <6>	 	06/04/97	rtm		removed call to QTVRUtils_IsQTVRMovie in InitApplicationWindowObject
//	   <5>	 	02/06/97	rtm		fixed window resizing code
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//	   
//////////

// header files
#include "ComApplication.h"

// header file for the specific test functions.
#include "TestFunctions.h"


// global variables for Macintosh code
#if TARGET_OS_MAC
Str255					gAppName = "\pVRCursors";				// the name of this application
#endif

// global variables for Windows code
#if TARGET_OS_WIN32
extern HWND				ghWnd;									// the MDI frame window; this window has the menu bar
extern int				gNumWindowsOpen;
extern LPSTR			gCmdLine;
#endif

long					gMaxMilliSecToUse = 0L;		
Boolean					gQTVRMgrIsPresent = false;				// is the QuickTime VR Manager available?		
long					gQTVRMgrVersion = 0L;					// the version of the QuickTime VR Manager	


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// before the MDI frame window is created or after. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {

		// make sure that the QuickTime VR Manager is available in the present operating environment;
		// if it is, get its version and (if necessary) initialize it
		if (QTVRUtils_IsQTVRMgrInstalled()) {
			gQTVRMgrIsPresent = true;
			gQTVRMgrVersion = QTVRUtils_GetQTVRVersion();		// get the version of the QuickTime VR Manager
		
#if TARGET_OS_WIN32
			// initialize the QuickTime VR Manager
			InitializeQTVR();									
#endif
		}
		
	}	// end of kInitAppPhase_BeforeCreateFrameWindow

	// do any start-up activities that should occur after the MDI frame window is created
	if (theStartPhase & kInitAppPhase_AfterCreateFrameWindow) {
		
#if TARGET_OS_WIN32
		// on Windows, open as movie documents any files specified on the command line
		DoOpenCommandLineMovies(gCmdLine);									
#endif

	}	// end of kInitAppPhase_AfterCreateFrameWindow
}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// before any open movie windows are destroyed or after.
//
//////////

void StopApplication (UInt32 theStopPhase)
{
	// do any shut-down activities that should occur after the movie windows are destroyed
	if (theStopPhase & kStopAppPhase_AfterDestroyWindows) {
	
#if TARGET_OS_WIN32
		// terminate QuickTime VR Manager
		if (gQTVRMgrIsPresent)
			TerminateQTVR();
#endif
			
	}	// end of kStopAppPhase_AfterDestroyWindows
}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif	// TARGET_OS_MAC
		}
	}

	// @@@INSERT APPLICATION-SPECIFIC IDLE-TIME FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	BeginUpdate(GetPortFromWindowReference(theWindow));
	
	// draw the movie controller and its movie
	MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, theWindow);
	
	EndUpdate(GetPortFromWindowReference(theWindow));
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// @@@INSERT APPLICATION-SPECIFIC CONTENT CLICKING FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean		isHandled = true;
	
	switch (theCharCode) {
	
		// @@@HANDLE APPLICATION-SPECIFIC KEY PRESSES HERE

		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
	WindowObject		myWindowObject = NULL;
	MovieController 	myMC = NULL;
	
	myWindowObject = GetWindowObjectFromFrontWindow();
	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;
	
	// make sure we have a valid movie controller
	if (myMC == NULL)
		return;
	
	switch (theMenuItem) {
		case IDM_CONTROLLER:
			QTUtils_ToggleControllerBar(myMC);
			break;

		case IDM_SPEAKER_BUTTON:
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRSpeakerButton))
				QTUtils_HideControllerButton(myMC, kQTVRSpeakerButton);
			else 
				QTUtils_ShowControllerButton(myMC, kQTVRSpeakerButton);
			break;

		case IDM_BACK_BUTTON:
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRBackButton))
				QTUtils_HideControllerButton(myMC, kQTVRBackButton);
			else 
				QTUtils_ShowControllerButton(myMC, kQTVRBackButton);
			break;

		case IDM_ZOOM_BUTTONS:
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRZoomButtons))
				QTUtils_HideControllerButton(myMC, kQTVRZoomButtons);
			else 
				QTUtils_ShowControllerButton(myMC, kQTVRZoomButtons);
			break;

		case IDM_HOTSPOT_BUTTON:
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRHotSpotButton))
				QTUtils_HideControllerButton(myMC, kQTVRHotSpotButton);
			else 
				QTUtils_ShowControllerButton(myMC, kQTVRHotSpotButton);
			break;

		case IDM_TRANSLATE_BUTTON:
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRTranslateButton))
				QTUtils_HideControllerButton(myMC, kQTVRTranslateButton);
			else 
				QTUtils_ShowControllerButton(myMC, kQTVRTranslateButton);
			break;
		
		default:
			break;
	}	// switch (theMenuItem)
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
// Currently, the Mac application has only one app-specific menu ("Test"); you could change that.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
	WindowObject		myWindowObject = NULL; 
	MovieController 	myMC = NULL;
	MenuReference		myMenu;
	
#if TARGET_OS_WIN32
	myMenu = theMenu;
#elif TARGET_OS_MAC
	myMenu = GetMenuHandle(kTestMenu);
#endif
	
	if (theWindow != NULL)
		myWindowObject = GetWindowObjectFromWindow(theWindow);

	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;

	// we don't allow creating new files here...
#if TARGET_OS_MAC
	SetMenuItemState(GetMenuHandle(mFile), iNew, kDisableMenuItem);
#endif

	if (myMC == NULL) {
		// no movie controller, so disable all the Test menu items
		SetMenuItemState(myMenu, IDM_CONTROLLER, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SPEAKER_BUTTON, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_BACK_BUTTON, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_ZOOM_BUTTONS, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_HOTSPOT_BUTTON, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_TRANSLATE_BUTTON, kDisableMenuItem);
	} else {
		// we've got a movie controller, so it's safe to proceed....
		SetMenuItemState(myMenu, IDM_CONTROLLER, kEnableMenuItem);

		// if controller bar is visible, enable Test menu items
		if (QTUtils_IsControllerBarVisible(myMC)) {
			SetMenuItemLabel(myMenu, IDM_CONTROLLER, "Hide &Controller");
       		// ungray the other menu itmes
			SetMenuItemState(myMenu, IDM_SPEAKER_BUTTON, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_BACK_BUTTON, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_ZOOM_BUTTONS, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_HOTSPOT_BUTTON, kEnableMenuItem);
			SetMenuItemState(myMenu, IDM_TRANSLATE_BUTTON, kEnableMenuItem);
			
			// handle QT and QTVR Test menu items
			if (QTUtils_IsControllerButtonVisible(myMC, kQTVRSpeakerButton))
				SetMenuItemLabel(myMenu, IDM_SPEAKER_BUTTON, "Hide Spea&ker Button");
			else 
				SetMenuItemLabel(myMenu, IDM_SPEAKER_BUTTON, "Show Spea&ker Button");

			// handle QTVR-specific Test menu items
			if ((**myWindowObject).fInstance != NULL) {

				if (QTUtils_IsControllerButtonVisible(myMC, kQTVRBackButton))
					SetMenuItemLabel(myMenu, IDM_BACK_BUTTON, "Hide &Back Button");
				else 
					SetMenuItemLabel(myMenu, IDM_BACK_BUTTON, "Show &Back Button");

				if (QTUtils_IsControllerButtonVisible(myMC, kQTVRZoomButtons))
					SetMenuItemLabel(myMenu, IDM_ZOOM_BUTTONS, "Hide &Zoom Buttons");
				else 
					SetMenuItemLabel(myMenu, IDM_ZOOM_BUTTONS, "Show &Zoom Buttons");

				if (QTUtils_IsControllerButtonVisible(myMC, kQTVRHotSpotButton))
					SetMenuItemLabel(myMenu, IDM_HOTSPOT_BUTTON, "Hide &Hot Spot Button");
				else 
					SetMenuItemLabel(myMenu, IDM_HOTSPOT_BUTTON, "Show &Hot Spot Button");

				if (QTUtils_IsControllerButtonVisible(myMC, kQTVRTranslateButton))
					SetMenuItemLabel(myMenu, IDM_TRANSLATE_BUTTON, "Hide &Translate Button");
				else 
					SetMenuItemLabel(myMenu, IDM_TRANSLATE_BUTTON, "Show &Translate Button");

			} else {	// we're not a QTVR movie, so disable QTVR-specific menus
				SetMenuItemState(myMenu, IDM_BACK_BUTTON, kDisableMenuItem);
				SetMenuItemState(myMenu, IDM_ZOOM_BUTTONS, kDisableMenuItem);
				SetMenuItemState(myMenu, IDM_HOTSPOT_BUTTON, kDisableMenuItem);
				SetMenuItemState(myMenu, IDM_TRANSLATE_BUTTON, kDisableMenuItem);
			}

			
		} else {
			// controller bar is not visible
			
			SetMenuItemLabel(myMenu, IDM_CONTROLLER, "Show &Controller");
      		// gray the other menu items
			SetMenuItemState(myMenu, IDM_SPEAKER_BUTTON, kDisableMenuItem);
			SetMenuItemState(myMenu, IDM_BACK_BUTTON, kDisableMenuItem);
			SetMenuItemState(myMenu, IDM_ZOOM_BUTTONS, kDisableMenuItem);
			SetMenuItemState(myMenu, IDM_HOTSPOT_BUTTON, kDisableMenuItem);
			SetMenuItemState(myMenu, IDM_TRANSLATE_BUTTON, kDisableMenuItem);
        }
	}
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
	return(false);			// no-op for now
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{
	Track						myQTVRTrack = NULL;
	Movie						myMovie = NULL;
	MovieController				myMC = NULL;
	QTVRInstance				myInstance = NULL;
	QTVRMouseOverHotSpotUPP		myInterceptProc;
		
	if (theWindowObject == NULL)
		return;

	// make sure we can safely call the QTVR API
	if (!gQTVRMgrIsPresent)
		return;

	// find the QTVR track, if there is one
	myMC = (**theWindowObject).fController;
	myMovie = (**theWindowObject).fMovie;
	myQTVRTrack = QTVRGetQTVRTrack(myMovie, 1);
	
	QTVRGetQTVRInstance(&myInstance, myQTVRTrack, myMC);
	(**theWindowObject).fInstance = myInstance;

	// do any QTVR window configuration
	if (myInstance != NULL) {
		
		// set unit to radians
		QTVRSetAngularUnits(myInstance, kQTVRRadians);

		// update
		QTVRUpdate(myInstance, kQTVRCurrentMode);
		
		// install a mouse-over hot spot procedure to change cursors for specific hot spot types
		myInterceptProc = NewQTVRMouseOverHotSpotProc(MyMouseOverHotSpotProc);
		QTVRSetMouseOverHotSpotProc(myInstance, myInterceptProc, 0, 0);
	}
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
	OSErr				myErr = noErr;
	QTVRInstance		myInstance = NULL;
		
	if (theWindowObject == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	
	// @@@INSERT APPLICATION-SPECIFIC WINDOW OBJECT CLEAN-UP HERE

	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	switch (theAction) {
	
		// handle window resizing
		case mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			break;

		// handle idle events
		case mcActionIdle:
			DoIdle((**myWindowObject).fWindow);
			break;
			
		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}